using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents string RPC parameters used in RPC request.
	/// </summary>
	public class RpcStringParameter : RpcParameter
	{
		private static readonly Type _baseType = typeof(string);

		// Constants used in XML serialization/deserialization
		private const string XMLCONSTS_PARAMETER_TYPE_NAME = "string";
		
		/// <summary>
		/// Shortcut constructor that does not initialize parameter value.
		/// </summary>
		/// <param name="paramPosition">RPC parameter position in remote procedure's signature.</param>	
		public RpcStringParameter( int paramPosition )
			: this( paramPosition, null ) {}

		/// <summary>
		/// Constructor initializing both parameter name and value.
		/// </summary>
		/// <param name="paramPosition">RPC parameter position in remote procedure's signature.</param>	
		/// <param name="paramValue">RPC parameter value.</param>
		public RpcStringParameter( int paramPosition, string paramValue )
			: base( XMLCONSTS_PARAMETER_TYPE_NAME, new ParameterPosition( paramPosition ), paramValue ) {}

		/// <summary>
		/// XML deserialization constructor creating parameter object from parameter XML element.
		/// </summary>
		/// <param name="parameterNode">XML node to parse and get parameter information from.</param>
		public RpcStringParameter( XmlNode parameterNode )
			: base( XMLCONSTS_PARAMETER_TYPE_NAME, parameterNode ) {}

		/// <summary>
		/// Constructor added to make this class extendable by derived classes. 
		/// Allows to specify different RPC parameter type. 
		/// </summary>
		/// <param name="paramTypeName">RPC parameter type name (allows derived classes override parameter type name).</param>
		/// <param name="paramPosition">RPC parameter position in remote procedure's signature.</param>	
		/// <param name="paramValue">RPC parameter value.</param>
		protected RpcStringParameter( string paramTypeName, ParameterPosition paramPosition, object paramValue )
			: base( paramTypeName, paramPosition, paramValue ) {}

		/// <summary>
		/// XML deserialization constructor added to make this class extendable by derived classes. 
		/// Allows to specify different RPC parameter type. 
		/// </summary>
		/// <param name="paramTypeName">RPC parameter type name (allows derived classes override parameter type name).</param>
		/// <param name="parameterNode">XML node to parse and get parameter information from.</param>
		protected RpcStringParameter( string paramTypeName, XmlNode parameterNode )
			: base( paramTypeName, parameterNode ) {}
			
		/// <summary>
		/// Xml serialization method writing out RPC parameter value.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteParameterValue( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteString( (string)BaseValue );
		}

		/// <summary>
		/// XML deserialization method reading parameter value.
		/// </summary>
		/// <param name="parameterElement">
		///		Parameter element. Parameter value resides in a child node of this element.
		///	</param>
		protected override object ParseParameterValue( XmlElement parameterElement )
		{
			if( parameterElement == null )
				throw( new ArgumentNullException( "parameterElement" ) );

			return parameterElement.InnerText;
		}

		/// <summary>
		/// Conversion operator overload added for ease of use.
		/// </summary>
		/// <param name="src">Source RPC parameter.</param>
		/// <returns>Parameter's string value.</returns>
		public static implicit operator string( RpcStringParameter src )
		{
			return (string)src.BaseValue;
		}

		/// <summary>
		/// This method is used during XML deserialization to recognize parameter XML.
		/// It tests supplied string for match with parameter type name and 
		/// returns true when there is a match.
		/// </summary>
		/// <param name="parameterTypeName">String to tests (expected to be a parameter type name).</param>
		/// <returns>True if supplied string matches parameter type name.</returns>
		public static bool DoesParameterTypeNameMatch( string parameterTypeName )
		{
			return parameterTypeName == XMLCONSTS_PARAMETER_TYPE_NAME;
		}

		/// <summary>
		/// Parameter value.
		/// </summary>
		public string Value
		{
			get
			{
				return (String)BaseValue;
			}
			set
			{
				SetBaseValue( value );
			}
		}

		/// <summary>
		/// RPC parameter's underlying .NET type. 
		/// </summary>
		public override Type BaseType
		{
			get
			{
				return _baseType;
			}
		}
	}
}
